# VECTOR MANAGEMENT - Update Server Setup Guide

## Server URL: https://victorhost.com/posupdate

## Required Files Structure on Server:

```
victorhost.com/posupdate/
├── version.json          # Contains latest version info
├── changelog.json        # Contains version history
└── updates/
    ├── update-1.0.1.zip  # Update package for v1.0.1
    ├── update-1.0.2.zip  # Update package for v1.0.2
    └── update-1.1.0.zip  # etc...
```

## 1. version.json
This file tells the system what the latest version is.

**File location:** `victorhost.com/posupdate/version.json`

**Content:**
```json
{
    "version": "1.0.1",
    "release_date": "2025-10-21",
    "minimum_php": "8.1",
    "minimum_laravel": "11.0"
}
```

**When you release v1.1.0, update to:**
```json
{
    "version": "1.1.0",
    "release_date": "2025-11-01",
    "minimum_php": "8.1",
    "minimum_laravel": "11.0"
}
```

---

## 2. changelog.json
This file contains the version history and what changed in each version.

**File location:** `victorhost.com/posupdate/changelog.json`

**Content:**
```json
[
    {
        "version": "1.1.0",
        "date": "2025-11-01",
        "changes": [
            "Added new reporting features",
            "Improved performance",
            "Fixed bugs in sales module",
            "Enhanced user interface"
        ]
    },
    {
        "version": "1.0.1",
        "date": "2025-10-21",
        "changes": [
            "Added enhanced notification system with sounds",
            "Improved sales history delete functionality",
            "Added auto-update system",
            "Bug fixes and performance improvements"
        ]
    },
    {
        "version": "1.0.0",
        "date": "2025-10-01",
        "changes": [
            "Initial release",
            "POS system with sales management",
            "Customer management",
            "Inventory tracking"
        ]
    }
]
```

---

## 3. Update Package Files
These are the actual update files that will be downloaded and installed.

**File location:** `victorhost.com/posupdate/updates/update-{version}.zip`

### How to Create Update Package:

1. **Make your code changes** in your development environment

2. **Create update ZIP** containing ONLY changed files:
   ```
   update-1.0.1.zip/
   ├── app/
   │   └── Http/
   │       └── Controllers/
   │           └── NewController.php
   ├── resources/
   │   └── views/
   │       └── new-view.blade.php
   ├── database/
   │   └── migrations/
   │       └── 2025_10_21_create_new_table.php
   └── UPDATE_INFO.txt  (optional - list of changes)
   ```

3. **Important:** 
   - Only include NEW or CHANGED files
   - Use same directory structure as your Laravel project
   - Don't include vendor/, node_modules/, or .env files
   - Include new migrations if database changes are needed

4. **Upload to server:**
   - Upload `update-1.0.1.zip` to `victorhost.com/posupdate/updates/`
   - Make sure file is publicly accessible
   - File should be downloadable via direct URL

---

## 4. Releasing a New Update

When you want to release a new update (e.g., v1.1.0):

### Step 1: Prepare Update Package
```bash
# Create ZIP with only changed files
zip -r update-1.1.0.zip app/ resources/ database/migrations/
```

### Step 2: Upload to Server
```bash
# Upload to victorhost.com/posupdate/updates/
scp update-1.1.0.zip user@victorhost.com:/path/to/posupdate/updates/
```

### Step 3: Update version.json
```json
{
    "version": "1.1.0",
    "release_date": "2025-11-01",
    "minimum_php": "8.1",
    "minimum_laravel": "11.0"
}
```

### Step 4: Update changelog.json
Add new version at the TOP of the array:
```json
[
    {
        "version": "1.1.0",
        "date": "2025-11-01",
        "changes": [
            "Your new features here",
            "Bug fixes",
            "etc..."
        ]
    },
    {
        "version": "1.0.1",
        ...
    }
]
```

### Step 5: Test Update
- Visit your POS system
- Click "VECTOR MANAGEMENT V1.0" in sidebar
- Should show "New Update Available! v1.1.0"
- Click "Start Update Now"
- System will automatically download and install

---

## 5. How It Works (User Side)

1. **User clicks** "VECTOR MANAGEMENT V1.0" in sidebar
2. **System checks** `victorhost.com/posupdate/version.json`
3. **Compares** current version (1.0.0) with latest (1.1.0)
4. **Shows update button** if new version available
5. **User clicks** "Start Update Now"
6. **System automatically:**
   - Creates backup
   - Downloads `update-1.1.0.zip` from server
   - Extracts files
   - Runs migrations
   - Clears cache
   - Shows success notification 🔊
7. **Done!** System updated to v1.1.0

---

## 6. Update System Configuration

The update system is configured in:
`app/Http/Controllers/SystemUpdateController.php`

```php
const CURRENT_VERSION = '1.0.0';  // Change this when you update
const UPDATE_SERVER_URL = 'https://victorhost.com/posupdate';
```

**After successful update, manually change CURRENT_VERSION in your code:**
```php
const CURRENT_VERSION = '1.1.0';  // Updated!
```

Then include this file in your next update package.

---

## 7. Testing Update System

### Test on Local Server First:
1. Create test update package
2. Upload to victorhost.com/posupdate/updates/
3. Update version.json to test version
4. Try update on test system
5. Verify everything works
6. Roll out to production

### Rollback if Needed:
System automatically creates backup before update. If something goes wrong, system will auto-rollback.

---

## 8. Security Recommendations

1. **Use HTTPS** for update server (already configured)
2. **Add authentication** to update endpoint (optional)
3. **Verify ZIP integrity** with checksums (future enhancement)
4. **Keep backups** of all update packages
5. **Test updates** on staging environment first

---

## 9. Quick Reference

| Action | File to Update |
|--------|----------------|
| New version available | version.json |
| What's new info | changelog.json |
| Actual update files | updates/update-X.X.X.zip |
| Current system version | SystemUpdateController.php |

---

## 10. Example Update Workflow

**Scenario: Releasing v1.2.0 with new features**

```bash
# 1. Develop features locally
# 2. Create update package
cd /path/to/your/project
zip -r update-1.2.0.zip app/Http/Controllers/NewFeatureController.php \
                         resources/views/new-feature/ \
                         database/migrations/2025_11_15_new_table.php

# 3. Upload to server
scp update-1.2.0.zip user@victorhost.com:/posupdate/updates/

# 4. Update version.json on server
echo '{"version":"1.2.0","release_date":"2025-11-15"}' > version.json

# 5. Update changelog.json on server
# (Add v1.2.0 entry to changelog)

# 6. Notify users or they will see it automatically when clicking sidebar
```

**Done!** All POS systems will now see update available! 🎉

---

## Support

For issues or questions:
- Email: support@victorhost.com
- Documentation: https://victorhost.com/posupdate/docs
